# -*- coding: utf-8 -*-

from odoo import models, fields, api, _
from odoo.addons import decimal_precision as dp
from odoo.exceptions import ValidationError

SETTLEMENT_TYPE = [
    ('prepaid', 'Prepaid'),
    ('pay', 'Cash on delivery'),
    ('monthly', 'Monthly'),
]

SHIPPING_TYPE = [
    ('express', 'Express delivery'),
    ('logistics', 'Logistics'),
    ('store', 'Store'),
    ('home', 'Home delivery'),
]


class SettlementType(models.Model):
    _name = 'sale.settlement.type'
    _rec_name = 'name'
    _description = 'Logistics settlement method'

    name = fields.Char(string='Name', required=True)
    code = fields.Char(string='Code')
    remark = fields.Text(string='Remark')
    active = fields.Boolean(string='Active', default=True)

    _sql_constraints = [
        ('name_uniq', 'unique(name)', 'Logistics settlement name already exists!')
    ]


class ShippingType(models.Model):
    _name = 'sale.shipping.type'
    _rec_name = 'name'
    _description = 'Shipping method'

    name = fields.Char(string='Name', required=True)
    code = fields.Char(string='Code')
    remark = fields.Text(string='Remark')
    active = fields.Boolean(string='Active', default=True)

    _sql_constraints = [
        ('name_uniq', 'unique(name)', 'Shipping method name already exists!')
    ]


PAYMENT_METHOD = [
    ('wxpay', 'WeChat Payment'),
    ('alipay', 'Alipay'),
    ('credit', 'Credit Card')
]


class SaleOrder(models.Model):
    _inherit = 'sale.order'

    settlement_type_id = fields.Many2one('sale.settlement.type', string='Logistics settlement method',
                                         domain=[('active', '=', True)],
                                         states={'sale': [('readonly', True)], 'done': [('readonly', True)]})
    shipping_type_id = fields.Many2one('sale.shipping.type', string='Shipping method',
                                       domain=[('active', '=', True)],
                                       states={'sale': [('readonly', True)], 'done': [('readonly', True)]})
    payment_method = fields.Selection(selection=PAYMENT_METHOD, string='Payment method',
                                      states={'sale': [('readonly', True)], 'done': [('readonly', True)]})
    insured_amount = fields.Float(string='Insured amount', digits=dp.get_precision('Product Price'), copy=False)
    payment_screenshot = fields.Binary(string='Payment voucher screenshot', attachment=True, copy=False)

    agent_shipper = fields.Char(string='Agent Shipper')
    agent_shipping_address = fields.Char(string='Agent Shipping Address')

    @api.multi
    def action_view_delivery(self):
        """
        This function returns an action that display existing delivery orders
        of given sales order ids. It can either be a in a list or in a form
        view, if there is only one delivery order to show.
        """
        if not self.payment_screenshot:
            msg = 'Please upload a screenshot of the payment voucher'
            raise ValidationError(_(msg))

        return super(SaleOrder, self).action_view_delivery()
